/*
 * Copyright 2014 Canonical Ltd.
 *
 * Authors:
 * Charles Kerr: charles.kerr@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <ctime>
#include <map>
#include <memory> // std::unique_ptr
#include <string>
#include <thread>

#include <glib.h>

#include <ubuntu/hardware/alarm.h>

#include "powerd-internal.h"
#include "log.h"

namespace
{

struct Request
{
    std::string name;
    time_t time;
};

std::map<std::string,Request> requests;

UHardwareAlarm hardware_alarm = nullptr;

// NB: only the main thread accesses 'worker'
std::unique_ptr<std::thread> worker;

void start_worker_thread();

void set_wakeup_time(time_t wakeup_time, const std::string& name)
{
    powerd_return_if_fail(hardware_alarm != nullptr);
    powerd_return_if_fail(wakeup_time >= time(nullptr));

    struct tm tm;
    char buf[32];
    localtime_r(&wakeup_time, &tm);
    strftime(buf, sizeof(buf), "%F %T", &tm);
    powerd_debug("setting hardware wakeup time to %s for %s", buf, name.c_str());

    struct timespec sleep_interval;
    sleep_interval.tv_sec = wakeup_time;
    sleep_interval.tv_nsec = 0;
    u_hardware_alarm_set_relative_to_with_behavior(
        hardware_alarm,
        U_HARDWARE_ALARM_TIME_REFERENCE_RTC,
        U_HARDWARE_ALARM_SLEEP_BEHAVIOR_WAKEUP_DEVICE,
        &sleep_interval);
}

void reset_alarm_clock()
{
    // find request that will occur next
    const time_t now = time(nullptr);
    const Request* next = nullptr;
    size_t n_left = 0;
    for (const auto& it : requests) {
        if (now < it.second.time) {
            ++n_left;
            if (!next || (it.second.time < next->time))
                next = &it.second;
        }
    }

    if (next != nullptr)
    {
        powerd_debug("%s found %zu remaining wakeup requests", G_STRFUNC, n_left); 
        set_wakeup_time(next->time, next->name);
        if (!worker)
            start_worker_thread();
    }
    else if (worker) // no pending requests...
    {
        powerd_debug("shortening wakeup_time so worker thread will finish soon");
        set_wakeup_time(time(nullptr)+1, G_STRFUNC);
    }
}

gboolean on_worker_thread_finished(gpointer /*unused*/)
{
    // tell the world that we woke up
    powerd_wakeup_signal_emit();

    // worker thread is done, so clean up our 'worker' variable
    powerd_debug("worker thread is finished working; calling join()");
    worker->join();
    worker.reset();
    powerd_debug("worker thread is destroyed.");

    reset_alarm_clock();

    return G_SOURCE_REMOVE;
}

void threadfunc(UHardwareAlarm hw_alarm)
{
    // wait for the next alarm
    powerd_debug("calling wait_for_next_alarm");
    UHardwareAlarmWaitResult wait_result;
    auto rc = u_hardware_alarm_wait_for_next_alarm(hw_alarm, &wait_result);
    powerd_debug("worker thread finished waiting for hw alarm");
    powerd_warn_if_fail(rc == U_STATUS_SUCCESS);
    u_hardware_alarm_unref(hw_alarm);

    // kick back to the main loop
    g_idle_add(on_worker_thread_finished, nullptr);
}

void start_worker_thread()
{
    powerd_return_if_fail(hardware_alarm != nullptr);

    powerd_debug("starting hardware alarm worker thread");
    u_hardware_alarm_ref(hardware_alarm);
    worker.reset(new std::thread(threadfunc, hardware_alarm));
}

} // private impl in unnamed namespace

void
wakeup_init(void)
{
    hardware_alarm = u_hardware_alarm_create();
}

void
wakeup_deinit(void)
{
    requests.clear();
    reset_alarm_clock();
    if (worker) {
        worker->join();
        worker.reset();
    }
    g_clear_pointer(&hardware_alarm, u_hardware_alarm_unref);
}

int
wakeup_request(const char* name, time_t wakeup_time, powerd_cookie_t cookie)
{
    powerd_return_val_if_fail(hardware_alarm, ENOTSUP);
    powerd_return_val_if_fail(wakeup_time > time(nullptr), EINVAL);
    powerd_return_val_if_fail(cookie != NULL, EINVAL);

    // generate a new uuid string
    uuid_t tmp;
    uuid_generate(tmp);
    uuid_unparse(tmp, cookie);

    requests[cookie] = { name, wakeup_time };

    reset_alarm_clock();
    return 0;
}

int
wakeup_clear(const powerd_cookie_t cookie)
{
    powerd_return_val_if_fail(hardware_alarm, ENOTSUP);
    auto it = requests.find(cookie);
    powerd_return_val_if_fail(it != requests.end(), EINVAL);

    requests.erase(it);

    reset_alarm_clock();
    return 0;
}

