#include "balancer"
#include "Dispatchers/tcpdispatcher/tcpdispatcher"
#include "Dispatchers/httpdispatcher/httpdispatcher"
#include "Dispatchers/udpdispatcher/udpdispatcher"

// #define SHOWDEBUG

void Balancer::serve() {
    int clsock = -1;

    // Start up wakeup/checkup handlers. These are always started - even
    // when config.wakeupsec() and config.checkupsec() are not defined
    // and have value 0. Via the web interface, the values can be later
    // changed, but we want to have the checkers running always.
    if (!config.foregroundmode() && config.sport()) {
	msg ("Starting wakeup thread.\n");
	Wakeupthread *wt = new Wakeupthread();
	if (!wt)
	    throw Error("Memory fault in Balancer::serve");
	wt->start();
	
	msg ("Starting checkup thread.\n");
	Checkupthread *ct = new Checkupthread();
	if (!ct)
	    throw Error("Memory fault in Balancer::serve");
	ct->start();
    }

    // Write the PID file.
    if (config.pidfile() != "") {
	FILE *f;
	if (! (f = fopen (config.pidfile().c_str(), "w")) )
	    throw Error(string("Cannot write pid file ") +
			config.pidfile() + ": " + strerror(errno));
	fprintf (f, "%u\n", getpid());
	fclose (f);
    }

    // Wait for activity, serve it.
    msg ((Mstr("Awaiting activity on fd ") + server_fd) + "\n");
    MEM(Memory::mem_mark("Balancer start"));
    MEM(Memory::mem_follow(true));
    while (true) {
	MEM(Memory::mem_display());
	Fdset fdset(0);
	fdset.add (server_fd);
	fdset.wait_r();

	if (! fdset.readable(server_fd)) {
	    // We caught a signal. Either a request to report status,
	    // or to terminate.
	    msg ("Interrupt seen\n");
	    if (terminate()) {
		msg ("Termination requested, XR will stop.\n");
		break;
	    } else if (report()) {
		msg ("Report requested\n");
		reportmsg ("*** XR STATUS REPORT STARTS ***\n");
		for (unsigned i = 0; i < nbackends(); i++) {
		    reportmsg("Back end " + backend(i).description() + "\n");
		    reportmsg((Mstr("  Status: ") +
			       backend(i).availablestr()) +
			      (Mstr(", ") + backend(i).livestr()) + "\n");
		    reportmsg((Mstr("  Connections: ") +
			       backend(i).connections())
			       +
			       (Mstr(" (max ") + backend(i).maxconn()) + "\n");
		    reportmsg((Mstr("  Served:") + backend(i).bytesserved()) +
			      (Mstr(" bytes, ") + backend(i).clientsserved()) +
			      " clients\n");
		}
		report (false);
		reportmsg ("*** XR STATUS REPORT ENDS ***\n");
		continue;
	    } else if (restart()) {
		msg ("Restart requested\n");
		config.restart();
	    } else {
		msg ("Non-meaningful interrupt or select timeout, resuming\n");
		continue;
	    }
	}

	// Got activity! Check total # of connections.
	msg ((Mstr("Got activity on fd ") + server_fd) + "\n");
	request_nr++;
	if (config.maxconn() && connections() >= config.maxconn()) {
	    msg ((Mstr("Not serving connection: already ") + connections()) +
		 (Mstr(" connection(s) (max ") + config.maxconn()) + ")\n");
	    continue;
	}

	if (server_fd) {
	    // In daemon mode (server_fd > 0): serve and loop again
	    struct sockaddr_in clname;
	    int size = sizeof(clname);
	    
	    // Accept the client if this is a TCP connection.
	    if (config.stype() != Servertype::t_udp) {
		if ( (clsock = accept (server_fd, (struct sockaddr *) &clname,
				       (socklen_t*) &size)) < 0 ) {
		    warnmsg(Mstr("Failed to accept network connection: ") +
			    Mstr(strerror(errno)) + "\n");		    
		    continue;
		}
		string clientip = inet_ntoa(clname.sin_addr);
		msg ((Mstr("Accepted connection from ") + clientip) +
		     (Mstr(" as client fd ") + clsock) + "\n");
	    }

	    // Show how we look
	    if (config.verbose()) {
		ostringstream o;
		msg ((Mstr("Balancer is serving ") + connections()) +
		     " clients\n");
		msg ("Current back end states:\n");
		for (unsigned i = 0; i < nbackends(); i++)
		    msg((Mstr("  Back end ") + backend(i).description()) +
			(Mstr(": ") + backend(i).connections()) +
			(Mstr(" connections, max ") + backend(i).maxconn()) +
			(Mstr(", status ") + backend(i).availablestr()) +
			(Mstr(", anticipated ") + IPStore::anticipated(i)) +
			"\n");
	    }

	    Dispatcher *d;
	    switch (config.stype()) {
	    case Servertype::t_tcp:
		d = new TcpDispatcher(clsock, clname.sin_addr);
		break;
	    case Servertype::t_http:
		d = new HttpDispatcher(clsock, clname.sin_addr);
		break;
	    case Servertype::t_udp:
		d = new UdpDispatcher(server_fd);
		break;
	    default:
		throw Error("Internal error, can't choose dispatcher");
		break;
	    }
	    if (!d)
		throw Error("Memory fault in Balancer::serve");

	    // Allocation boundary printout
	    if (config.debug()) {
		void *mem = malloc(16);
		free (mem);
		_debugmsg (Mstr("Allocation boundary at dispatcher start: ") +
			  mem + "\n");
	    }
	    #ifdef SHOWDEBUG
	    void *mem = malloc(16);
	    free (mem);
	    cout << "XR allocation at dispatcher start: " << mem << '\n';
	    #endif

	    d->start();
	} else {
	    // If fd-serving, serve and close. Don't thread it up.
	    TcpDispatcher *d;
	    struct in_addr dummy;
	    inet_aton ("0.0.0.0", &dummy);

	    switch (config.stype()) {
	    case Servertype::t_tcp:
		d = new TcpDispatcher (server_fd, dummy);
		break;
	    case Servertype::t_http:
		d = new HttpDispatcher (server_fd, dummy);
		break;
	    case Servertype::t_udp:
		throw Error("UDP dispatching not allowed in inetd-mode");
		break;
	    default:
		throw Error("Internal error, can't choose dispatcher");
		break;
	    }
	    if (!d)
		throw Error("Memory fault in Balancer::serve");
	    d->execute();
	    break;
	}

	// If we exceed the max # of requests, stop..
	if (config.quitafter()) {
	    msg ((Mstr("Request ") + requestnr()) +
		 (Mstr(" underway of max ") + config.quitafter()) + "\n");
	    if (requestnr() >=  (long)config.quitafter()) {
		msg ("Max requests served, will stop.\n");
		break;
	    }
	}
    }

    // We're stopping now. If a PID stamp was created, remove it.
    if (config.pidfile() != "")
	unlink (config.pidfile().c_str());

    // Wait for running threads to die off.
    socketclose (server_fd);
    delete webinterface;
    unsigned prev_conn = 0x19081962;
    while (1) {
	unsigned curr_conn = balancer.connections();
	if (!curr_conn)
	    break;
	if (curr_conn != prev_conn) {
	    msg ((Mstr("There are still ") + curr_conn) + " connections\n");
	    prev_conn = curr_conn;
	}
	sleep (1);
    }
    msg ("XR is idle, stopping.\n");
}
