/* This file is part of lomiri-action-api
 * Copyright 2013 Canonical Ltd.
 *
 * lomiri-action-api is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * lomiri-action-api is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0

/*!

\qmltype Action
\inqmlmodule Lomiri.Action 1.0
\ingroup action-base
\since 1.0
\brief The main action class.

Lomiri services visualizing this class will usually
be represented it as a simple button or menu item, depending upon where it is contributed.

The optional name property is available through D-Bus and can be used to activate a specific Action
from external componenets such as the Launcher. See \l {Platform Integration} and
\l {Offline Actions} for more information.

If the parameterType property is set, the Action is said to be parameterised. This means that when it is
bound to a menu or button, the action expects a typed input parameter. The type affects the allowed value
of the QVariant that must be passed to the trigger() and triggered().

\b{note:} As QVariant does automatic conversions between different normally uncorvertable types the developer
      must be careful of the side effects the conversions might have if accidentally passing wrong type
      of a parameter to trigger() or when handling the value of triggered(). Please, see the QVariant
      documentation for more information on the conversions QVariant does.

Action has to be added to the ActionManager or a ActionContext to make it available for external components.

\qml
import Lomiri.Action 1.0

Action {
    id: myaction1
    name: "myaction1"
    text: i18n.tr("My Action")
    description: i18n.tr("Perform foo.")
    onTriggered: {
        console.debug ("myaction1 is triggered.");
    }
}

\endqml

*/


Item {
    /*!
      \qmlproperty string Action::name
      The name of the action. By default an action gets it's name
      generated automatically if not overridden with later.
      If name is set to "" then the action restores it's autogenerated name.

      The actions is accessible from D-Bus with this name.

      The name is not user visible.

      \b {note:} Changing the name is potentially an expensive operation if the action is already added to the manager.
                 If possible, set the name for your action before adding it to the manager.
    */
    property string name

    /*!
      \qmlproperty string Action::text : ""
      The user visible primary label of the action.
    */
    property string text : ""

    /*!
      \qmlproperty string Action::description : ""
      User visible secondary description for the action.

      Description is more verbose than the text and should describe the Action with
      couple of words.
    */
    property string description : ""

    /*!
      \qmlproperty string Action::keywords
      Additional user visible keywords for the action.

      Keywords improve the HUD search results when the user tries to search for an action
      with a synonym of the text. For example if we the application has an action "Crop"
      but the user tries to "Trim" then without the keywords the HUD would not try to offer
      the "Crop" action.

      The format of the keywords string is "Keyword 1;Keyword 2;Keyword 3" to allow translators
      to define different number of keywords per language.

      The keywords are separated by ; and they may contain spaces.

      \qml
      Action {
        text: i18n.tr("Crop")
        description: i18n.tr("Crop the image")
        keywords: i18n.tr("Trim;Cut")
      }
      \endqml
    */
    property string keywords

    /*!
      \qmlproperty bool Action::enabled : true

      If set to false the action can not be triggered.

      Components visualizing the action migth either hide the
      action or make it insensitive.

      If set to false the Action is removed from the search
      results of the HUD.
    */
    property bool enabled : true

    /*!
      \qmlproperty string Action::iconName : ""

      Name of a icon for this action.

      When the action is exported to external components the iconName
      must be avaible on system icon theme engine.
     */
    property string iconName : ""

    /*!
      \qmlproperty Type Action::parameterType : None

      Type of the parameter passed to trigger() and triggered().

      \e {Type} is an enumeration:
      \list
       \li \b{Action.None}: No paramater. (default)
       \li \b{Action.String}: String parameter.
       \li \b{Action.Integer}: Integer parameter.
       \li \b{Action.Bool}: Boolean parameter.
       \li \b{Action.Real}: Single precision floating point parameter.
      \endlist

      \qml
        Action {
            id: action
            parameterType: Action.String
            onTriggered: {
                // value arguments now contain strings
                console.log(value);
            }
        Component.onCompleted: action.trigger("Hello World")
      \endqml

      \b {note:} Changing the parameterType is potentially an expensive operation if the action is already added to the manager.
      If possible, set the parameterType of your action before adding it to the manager.
     */
    property Type parameterType : None

    /*!
     \qmlsignal Action::triggered(var value)
     The value is always compatible with the set parameterType.

    */
    signal triggered(var value)

    /*!
      \qmlmethod void Action::trigger(var value)

      Checks the value agains parameterType and triggers the action.

      If paramType is Action::None the action can be triggered by simly calling:
      \qml
        action.trigger();
      \endqml
    */
    function trigger(value) {}
}


