package jalview.bin.argparser;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import jalview.bin.Cache;
import jalview.bin.Console;
import jalview.datamodel.AlignmentI;
import jalview.datamodel.PDBEntry;
import jalview.datamodel.SequenceI;
import jalview.gui.AssociatePdbFileWithSeq;
import jalview.gui.Desktop;
import jalview.gui.StructureChooser;
import jalview.gui.StructureViewer.ViewerType;
import jalview.io.AppletFormatAdapter;
import jalview.io.DataSourceType;

public class StructuresFile
{
  private boolean superpose = Cache
          .getDefault(StructureChooser.AUTOSUPERIMPOSE, true);

  private String viewerid = null;

  private ViewerType viewerType = ViewerType.JMOL;

  private List<StructureLine> structureLines;

  private List<SequenceI> seqs = new ArrayList<>();

  private List<PDBEntry> pdbs = new ArrayList<>();

  private AlignmentI alignment;

  public StructuresFile(AlignmentI al, String filename)
  {
    alignment = al;
    setStructureLines(parseStructuresFile(new File(filename)));
  }

  public boolean isSuperpose()
  {
    return superpose;
  }

  public void setSuperpose(boolean superpose)
  {
    this.superpose = superpose;
  }

  public String getViewerid()
  {
    return viewerid;
  }

  public void setViewerid(String viewerid)
  {
    this.viewerid = viewerid;
  }

  public ViewerType getViewerType()
  {
    return viewerType;
  }

  public void setViewerType(ViewerType viewerType)
  {
    this.viewerType = viewerType;
  }

  public List<StructureLine> getStructureLines()
  {
    return structureLines;
  }

  public void setStructureLines(List<StructureLine> structureLines)
  {
    this.structureLines = structureLines;
  }

  public List<StructureLine> parseStructuresFile(File structuresFile)
  {
    if (!structuresFile.exists())
    {
      String message = Arg.STRUCTURESFILE.argString() + ArgParser.EQUALS
              + "\"" + structuresFile.getPath()
              + "\": File does not exist.";
      Console.warn(message);
      return null;
    }
    List<String> lines = ArgParser.readArgFile(structuresFile);
    List<StructureLine> structurelines = new ArrayList<>();
    if (lines != null)
    {
      for (String line : lines)
      {
        if (line.startsWith(ArgParser.DOUBLEDASH))
        {
          String val = getArgVal(Arg.VIEWERID, line);
          if (val != null)
          {
            setViewerid(val);
            continue;
          }
          val = getArgVal(Arg.STRUCTUREVIEWER, line);
          if (val != null)
          {
            setViewerType(ViewerType.getFromString(val));
            continue;
          }
          Boolean b = getArgBoolean(Arg.SUPERPOSE, line);
          if (b != null)
          {
            setSuperpose(b);
            continue;
          }
        }
        StructureLine sl = new StructureLine(line);
        structurelines.add(sl);
        String structureLocationRef = sl.getSource();
        DataSourceType structureLocationType = AppletFormatAdapter
                .checkProtocol(structureLocationRef);
        if (DataSourceType.FILE.equals(structureLocationType))
        {
          structureLocationRef = new File(structureLocationRef)
                  .getAbsolutePath();
        }
        SequenceI seq = alignment.findName(sl.getSeqid());

        // TODO get global or subval values for TFT and notempfac

        PDBEntry pdbEntry = new AssociatePdbFileWithSeq()
                .associatePdbWithSeq(structureLocationRef,
                        structureLocationType, seq, false, Desktop.instance,
                        null, null, false);

        pdbs.add(pdbEntry);
        seqs.add(seq);

      }
    }
    return structurelines;
  }

  public SequenceI[] getSeqs()
  {
    SequenceI[] array = new SequenceI[seqs.size()];
    return seqs.toArray(array);
  }

  public PDBEntry[] getPdbEntries()
  {
    PDBEntry[] array = new PDBEntry[pdbs.size()];
    return pdbs.toArray(array);
  }

  private static String getArgVal(Arg a, String line)
  {
    String s = new StringBuilder(a.argString()).append(ArgParser.EQUALS)
            .toString();
    if (line.startsWith(s))
    {
      return line.substring(s.length());
    }
    return null;
  }

  private static Boolean getArgBoolean(Arg a, String line)
  {
    if (line.equals(a.argString()))
    {
      return true;
    }
    if (line.equals(a.negateArgString()))
    {
      return false;
    }
    return null;
  }

}
