/*
 * IRiver ifp supporting functions
 * $Id: readwrite.c,v 1.2 2005/08/25 04:10:54 jim-campbell Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

//This needs a home.  (Currently declared in ifp_prim.h)
int ifp_copy_parent_string(char * p, const char * f, int n) {
	int i = 0;
	char * c;

        if (n){} // satisfy compiler
        c = strrchr(f, '\\');
	if (c == NULL) {
		ifp_err("apparently '%s' doesn't contain a '\\'\n", f);
		i = -1;
		goto out;
	}
	if (c == f) {
		//special case
		p[0] = '\\';
		p[1] = '\0';
	} else {
		memcpy(p, f, (c-f));
		p[c-f] = '\0';
	}

out:
	return i;
}

/** returns -EACCES */
static int check_permissions(struct ifp_device * dev, const char * f) {
	//ifp_dbg("here in check_perms");
	if (dev->model <= IFP_5XX) {
		const char * s;
		int n = strlen(f);
		if (n <= 4) {
			return 0;
		}
		s = f + n - 4;
		if (ifp_strnicmp(s, ".mp3", 4) == 0
			|| ifp_strnicmp(s, ".wma", 4) == 0
			|| ifp_strnicmp(s, ".asf", 4) == 0)
		{
			return -EACCES;
		}
	}

	return 0;
}

//copies 's' to 'd' and swaps the last two letters of the filename.
//Eg. ".mp3" becomes ".m3p".. etc.
static int mangle_filename(char * d, const char * s, int n)
{
	int l = strlen(s);

	strncpy(d, s, n);

	d[l-1] = s[l-2];
	d[l-2] = s[l-1];

	return 0;
}


/** \brief Opens the file f for reading.
 *
 * Returns -ENOENT if 'f' doesn't exist and -EACCES if 'f' is read-protected
 * by the device. */
int ifp_read_open(struct ifp_device * dev, const char * f)
{
	const char * realfile = NULL;
	int i = 0;
	int j;
	//ifp_dbg("here in ifp_read_open");
	if (dev->mode != IFP_MODE_NONE) {
		ifp_err("device has been left for %sing.",
			dev->mode == IFP_MODE_READING ? "read" : "writ");
		i = -1;
                goto out;
	}
        strncpy((char*)dev->filename, f, IFP_BUFFER_SIZE);
        i = ifp_copy_parent_string((char*)dev->dirname, f, IFP_BUFFER_SIZE);
	ifp_err_jump(i, out, "error copying directory name");

        i = check_permissions(dev, (char*)dev->filename);
	if (i == -EACCES) {
		//rename file
                i = mangle_filename((char*)dev->iobuff, (char*)dev->filename, IFP_BUFFER_SIZE);
		ifp_err_jump(i, out, "mangle failed for '%s'", dev->filename);

                i = ifp_rename(dev, (char*)dev->filename, (char*)dev->iobuff);
		ifp_err_jump(i, out, "rename from %s to %s failed",
			dev->filename, dev->iobuff);
                realfile = (char*)dev->iobuff;
	} else {
		ifp_err_jump(i, out, "filename permission check failed for '%s'",
			dev->filename);
                realfile = (char*)dev->filename;
	}


        i = ifp_dir_open(dev, (char*)dev->dirname);
	ifp_err_expect(i, i==-ENOENT, out,
		"error opening directory '%s'.", dev->dirname);

	i = ifp_file_open(dev, realfile);
	ifp_err_expect(i, i==-ENOENT, out2,
		"error opening file '%s'.", realfile);

	dev->mode = IFP_MODE_READING;
	dev->current_offset = 0;
	dev->filesize = ifp_file_size(dev);
	if (dev->filesize < 0) {
		ifp_err_i((int)(dev->filesize), "error reading filesize of '%s'.", f);
		goto out3;
	}
	dev->readcount = 0;
	dev->alt_readcount = 0;

	//ifp_dbg("opened %s successfully, returning %d", f, i);

	return i;
out3:
	j = ifp_file_close(dev);
	ifp_err_jump(j, out2, "file close also failed");
out2:
	j = ifp_dir_close(dev);
	ifp_err_jump(j, out1, "dir close also failed");
out1:
	dev->mode = IFP_MODE_NONE;
out:
	return i;
}
IFP_EXPORT(ifp_read_open);

/** \brief Closes a file open for reading.  */
int ifp_read_close(struct ifp_device * dev)
{
	int i = 0;

	if (dev->alt_readcount != dev->readcount) {
		ifp_err("readcounts don't match.  readcount=%d, alt_readcount=%d",
			dev->readcount, dev->alt_readcount);
	}

	i = ifp_file_close(dev);
	if (i) {
		ifp_err_i(i, "file close failed");
	}
	i = ifp_dir_close(dev);
	if (i) {
		ifp_err_i(i, "dir close failed");
	}

        i = check_permissions(dev, (char*)dev->filename);
	if (i == -EACCES) {
                i = mangle_filename((char*)dev->iobuff, (char*)dev->filename, IFP_BUFFER_SIZE);
		ifp_err_jump(i, out, "mangle failed for '%s'", dev->filename);
                i = ifp_rename(dev, (char*)dev->iobuff, (char*)dev->filename);
		ifp_err_jump(i, out, "rename from %s to %s failed",
			dev->iobuff, dev->filename);
	} else if (i != 0) {
		ifp_err_i(i, "filename permission check failed for '%s'",
			dev->filename);
	}

	dev->mode = IFP_MODE_NONE;
	//ifp_dbg("closing %s, returning %d", dev->filename, i);
out:

	return i;
}
IFP_EXPORT(ifp_read_close);

//Reads next block.  'bytes' should be IFP_BULK_BUFF_SIZE unless
//it's the last block--in which case 'bytes' is the number of bytes
//we believe are left to download.
static int read_next_block(struct ifp_device * dev, int bytes) {
	int i;
	IFP_BUG_ON(bytes > IFP_BULK_BUFF_SIZE);
	if (dev->readcount * IFP_BULK_BUFF_SIZE + bytes > dev->filesize) {
		ifp_err("Sanity check failed.  We've read %d x 16384 bytes, and"
			" are about to read %d more from a %d byte file.",
			dev->readcount, bytes, (int)dev->filesize);
	}
	i = ifp_file_download(dev, dev->iobuff, IFP_BULK_BUFF_SIZE);
	if (i < 0) {
		ifp_err_i(i, "error reading block at %s+%#lx",
			dev->filename, (long)dev->current_offset);
		goto out;
	} else if (i != bytes) {
		dev->readcount++;
		if (dev->alt_readcount != dev->readcount) {
			ifp_err("readcount=%d, alt_readcount=%d",
				dev->readcount, dev->alt_readcount);
		}
		if (dev->download_pipe_errors) {
			//ifp_dbg("pipe error hit, with corruption: epipes=%d, readcount=%d, bytes received=%d, bytes expected=%d.",
			//dev->download_pipe_errors, dev->readcount, i, bytes);
		} else {
			ifp_err("error reading block.. I expected %d bytes but got %d; readcount is %d",
				bytes, i, dev->readcount);
		}
		i = -EIO;
		goto out;
	} 
	dev->readcount++;
	IFP_BUG_ON(i > bytes);
	i = 0;

	//not strickly necessary..
	if (bytes < IFP_BULK_BUFF_SIZE) {
		memset(dev->iobuff + bytes, 0, IFP_BULK_BUFF_SIZE - bytes);
	}

out:
	return i;
}

/** \brief Reads from an open file.
  Reads the next 'bytes' of data into 'buff'.
  \return the number of bytes read, or <0 on error.
 */
int ifp_read_data(struct ifp_device * dev, void * buff, int bytes)
{
	int bytes_requested = bytes;
	int bytes_read = 0;
	int n;
	int i = 0;
	int block_off;
	int available;
	uint8_t * o = buff;

	bytes = min(bytes, (int)(dev->filesize - dev->current_offset));
	while (bytes > 0) {
		block_off = dev->current_offset % IFP_BULK_BUFF_SIZE;
		if (block_off == 0) {
			//read next block
			available = min(IFP_BULK_BUFF_SIZE,
				(int)(dev->filesize - dev->current_offset));

			i = read_next_block(dev, available);
			if (i) {
				//quiet on pipe errors
				if (!(i == -EIO && dev->download_pipe_errors > 0)) {
					ifp_err_i(i, "error reading next block"
					" filesize=%d position=%d", (int)dev->filesize,
					(int)dev->current_offset);
				}
				goto out;
			}
		} else {
			available = IFP_BULK_BUFF_SIZE - block_off;
		}
		n = min(bytes, available);
		IFP_BUG_ON(n <= 0);

		memcpy(o, dev->iobuff + block_off, n);
		o += n;
		dev->current_offset += n;
		bytes -= n;
		bytes_read += n;
	}
	IFP_BUG_ON(bytes < 0);
	IFP_BUG_ON(bytes != 0);

	if (bytes_read != bytes_requested && dev->current_offset != dev->filesize) {
		ifp_dbg("returning %d instead of %d (but not EOF)", bytes_read, bytes_requested);
	}
	IFP_BUG_ON(bytes_read != bytes_requested && dev->current_offset != dev->filesize);

	return bytes_read;
out:
	return i;
}
IFP_EXPORT(ifp_read_data);

/** \brief Check if we've reached the end of the file.  (EOF) */
int ifp_read_eof(struct ifp_device * dev) {
	return dev->current_offset == dev->filesize;
}
IFP_EXPORT(ifp_read_eof);

/** \brief Returns the size of the current file in bytes. */
int ifp_read_size(struct ifp_device * dev) {
	return dev->filesize;
}
IFP_EXPORT(ifp_read_size);

static int fake_block_reads(struct ifp_device * dev, int n) {
	int actual, j,i = 0;
	const int blocksize = IFP_BULK_BUFF_SIZE;
	ifp_dbg("here -- sanity check");
	i = _ifp_set_buffer_size(dev, blocksize, 1);
	if (i) {
		ifp_err_i(i, "set buffer failed");
		return i > 0 ? -EIO : i;
	}
        for (j=0; j < n; j++) {
                i = _ifp_file_download(dev, (int)blocksize, &actual);
		ifp_err_jump(i, out, "download control message failed");
		dev->readcount++;

                if (actual != blocksize) {
                        i = -EIO;
                        ifp_err("fake read failed,  I can't handle "
                                "getting %d bytes instead of %d\n",
				actual, (int)blocksize);
			goto out;
                }
        }

out:
	return i;
}

/** \brief Fast-forward within the current file.
 *
 * In the current open file, skip forward to 'bytes' (ignoring the data).
 * Caution: the implementation isn't particularily fast, and can only seek
 * forward.  Avoid it if you can.
 */
int ifp_read_seek(struct ifp_device * dev, int bytes)
{
        int i = 0;
	int available = 0;

        const int blocksize = IFP_BULK_BUFF_SIZE;
        //int bitsize   = IFP_BULK_BUFF_BITS;
        ifp_off_t tar_offset = dev->current_offset + bytes;
        ifp_off_t cur_offset = dev->current_offset;
        int cur_block =  cur_offset == 0 ? -1 : (int)(cur_offset-1) / blocksize;
        int tar_block =  tar_offset == 0 ? -1 : (int)(tar_offset-1) / blocksize;
        int count     = tar_block - cur_block;

	if (tar_block != -1) {
		ifp_dbg("seeking forward %d blocks from %ld to %ld (%d:%d to %d:%d)",
			count, (long)cur_offset, (long)tar_offset,
			cur_block, (int)cur_offset % blocksize,
			tar_block, (int)tar_offset % blocksize);
	}
	//ifp_dbg("seeking forward %d blocks from %ld to %ld (%d:%d to %d:%d)",
	//	count,
	//	(long)cur_offset, (long)tar_offset,
	//	cur_block, (int)cur_offset % blocksize,
	//	tar_block, (int)tar_offset % blocksize);
	IFP_BUG_ON(count < 0);

	//cases:
	//	count == 0	update current_offset and return
	//	count == 1	download next block, update current_offset
	//	count >= 2	several fake downloads, ...
	switch (count > 1 ? 2 : count) {
	case 2:
		//ifp_dbg("doing fake reads");
		i = fake_block_reads(dev, count - 1);
		ifp_err_jump(i, out, "fake block reads failed");
		//fallthrough
	case 1:
		//ifp_dbg("doing real read");
		available = (int)(dev->filesize - (tar_block*blocksize));
        if (available > blocksize) {
            available = blocksize;
        }
		i = read_next_block(dev, available);
		//ifp_err("filesize=%d, current_pos=%d; bytes=%d; available=%d",
		//	(int)dev->filesize, (int)dev->current_offset,
		//	bytes, available);
		ifp_err_jump(i, out, "error reading destination block of %d bytes",
			available);
		//fallthrough
	case 0:
		//ifp_dbg("adjusting 'current_offset' by %d", bytes);
        	dev->current_offset += bytes;
		break;
	default:
		IFP_BUG_ON(1);
	}
	//ifp_dbg("done");
	if (i) {
		ifp_dbg("returning %d", i);
	}
out:
        return i;
}
IFP_EXPORT(ifp_read_seek);

/*This code is borrowed from ifp-line and appears to work fine.  The original
 *driver behaves slightly differently:
 *
 * on open
 *	ifp_dir_open(d)
 *	ifp_file_open(f)
 *		fails because file doesn't exist yet.  Good.
 *	ifp_dir_close()
 *
 *	ifp_dir_open(d)
 *	ifp_file_open_new(d)
 *
 * on close
 *	ifp_file_close()
 *	ifp_dir_close()
 *
 *	ifp_freespace()
 *
 *	ls, I think.
 */

/** \brief Opens the file f for writing.
 *
 *  Creates and opens a new file of 'filesize' bytes with the name 'f'.
 * Returns -EEXIST if the name 'f' isn't available.  (Ie,
 * if there allready exists a file or directory with the same name.)
 * Returns -ENOENT, -ENOSPC, or IFP_ERR_BAD_FILENAME
 */
int ifp_write_open(struct ifp_device * dev, const char * f, int filesize)
{
	int i;
	int j;

	if (dev->mode != IFP_MODE_NONE) {
		ifp_err("device has been left for %sing.",
			dev->mode == IFP_MODE_READING ? "read" : "writ");
		i = -1;
		goto out;
	}

	memcpy(dev->filename, f, IFP_BUFFER_SIZE);

        i = ifp_copy_parent_string((char*)dev->dirname, f, IFP_BUFFER_SIZE);
	ifp_err_jump(i, out, "error copying directory name");

        i = ifp_dir_open(dev, (char*)dev->dirname);
	ifp_err_expect(i, i==-ENOENT, out, "open_dir request failed.");

	i = ifp_freespace(dev);
	if (i < 0) {
		ifp_err_jump(i, out2, "free space request failed");
	} else if (i < filesize) {
		i = -ENOSPC;
		ifp_err_jump(i, out2, "not enough free space on the device");
	}

        i = ifp_file_open_new(dev, (char*)dev->filename, filesize);
	ifp_err_expect(i, i==-EEXIST||i==IFP_ERR_BAD_FILENAME, out2, "file create failed");

	dev->mode = IFP_MODE_WRITING;
	dev->current_offset = 0;
	dev->filesize = filesize;
	dev->readcount = 0;

	return i;
out2:
	j = ifp_dir_close(dev);
	ifp_err_jump(j, out1, "dir close also failed");
out1:
	dev->mode = IFP_MODE_NONE;
out:
	return i;
}
IFP_EXPORT(ifp_write_open);

static int quick_write_verify(struct ifp_device * dev) {
	int i = 0;
	int size;

        i = ifp_dir_open(dev, (char*)dev->dirname);
	ifp_err_jump(i, out, "open dir failed");

        i = ifp_file_open(dev, (char*)dev->filename);
	ifp_err_jump(i, out, "open file failed");

	size = ifp_file_size(dev);
	if (size < 0) {
		ifp_err_i(size, "file size query failed");
		goto out;
	}

	i = ifp_file_close(dev);
	ifp_err_jump(i, out, "close file failed");

	i = ifp_dir_close(dev);
	ifp_err_jump(i, out, "close dir failed");

	if (size != dev->current_offset) {
		ifp_err("reported file size is %d instead of %d.. upload failed",
			size, (int)dev->filesize);
		i = -EIO;
	}

out:
	return i;
}

/** \brief Closes a file open for writing. */
int ifp_write_close(struct ifp_device * dev)
{
	int i = 0;
	int e = 0;
	int remainder = dev->current_offset % IFP_BULK_BUFF_SIZE;
	if (remainder != 0) {
		i = ifp_file_upload(dev, dev->iobuff, remainder);
		if (i != remainder) {
			ifp_err_i(i, "problem uploading last %d bytes.  "
			"Attempting to close file anyways.", remainder);
			e=e?e:i;
			//no jump
		}
	}
	
#if 0
	//User can abort transfers, but we still have to close with grace
	ifp_wrn_on(dev->current_offset != dev->filesize,
		"received %d fewer bytes than expected",
		(int)(dev->filesize - dev->current_offset));
#endif

	//ifp_dbg("new write code finishing for %s.. all good, apparently.",
	//	dev->filename);

	i = ifp_file_flush(dev);
	if (i) {
		ifp_err_i(i, "flush failed, closing anyways.");
		e=e?e:i;
		//no jump
	}

	i = ifp_file_close(dev);
	if (i) {
		ifp_err_i(i, "file close failed, closing dir anyways.");
		e=e?e:i;
		//no jump
	}
	i = ifp_dir_close(dev);
	if (i) {
		ifp_err_i(i, "dir close failed.");
		e=e?e:i;
	}

	if (e == 0) {
		i = quick_write_verify(dev);
		if (i) {
			ifp_err_i(i, "quick-verify failed--upload may have failed.");
		}
	}
	dev->mode = IFP_MODE_NONE;

	return i?i:e;
}
IFP_EXPORT(ifp_write_close);

/** \brief Writes 'bytes' of data from buff to the file.
 *
 * Returns 0 on success.  (Does not return the number of bytes written--it's
 * all or nothing.)
 */
int ifp_write_data(struct ifp_device * dev, void * buff, int bytes)
{
	int e = 0;
	uint8_t * i = buff;
	uint8_t * o = dev->iobuff;
	int n;
	const int bs = IFP_BULK_BUFF_SIZE;
	int block_off = dev->current_offset % bs;
	while (bytes > 0) {
		n = bs - block_off;
		if (n > bytes) {
			n = bytes;
		}
		memcpy(o + block_off, i, n);
		{
			//ifp_dbg("ifp_write_data: block_off is %d, bytes to write is %d.", block_off, n);
		}

		i += n;
		block_off += n;
		bytes -= n;
		dev->current_offset += n;

		if (block_off == bs) {
			//At the end of the file, we have to write a partial block
			e = ifp_file_upload(dev, o, block_off);
			if (e == 0) {
                                ifp_dbg("ifp_file_upload returned 0: dev->current_offset = %ld", dev->current_offset);
			}
			else if (e != block_off) {
				ifp_err_jump(e, out, "upload of %d bytes failed",
					block_off);
				block_off = 0;
			}
			e = 0;
		}
	}
out:
	return e;
}
IFP_EXPORT(ifp_write_data);

